import ij.IJ;
import ij.ImagePlus;
import ij.Macro;
import ij.WindowManager;
import ij.gui.GenericDialog;
import ij.plugin.PlugIn;
import ij.process.ColorProcessor;
import ij.process.ImageProcessor;

import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.util.*;

import com.github.sarxos.webcam.Webcam;

import ij.measure.Calibration;
import ij.Prefs;

/**
 * @author Jerome Mutterer
 * @date 2014 12 15 first release
 * @date 2015 01 14 added macro support and custom resolutions thanks to Jarom
 *       Jackson
 * @date 2015 02 03 added proper macro support using GenericDialog for options
 *       thanks to Wayne Rasband
 * @modif François Gannier
 * @date 2015 06 01 added calibration
 * @date 2015 06 01 added saving prefs
 * @date 2015 06 21 added fps on statusbar
 * @date 2015 06 24 fixed error with no webcam plugged
 * 
 */

public class IJ_webcam_plugin implements PlugIn {

	Webcam camera;
	BufferedImage image;
	ImagePlus imp, imp2;
	ImageProcessor ip;
	int camID = 0, width = 0, height = 0;
	float calib = 1;
	String unit = "pixel";
	boolean grab = false, customSize = false;
	boolean displayFPS = true;

	public void run(String s) {

		camera = Webcam.getDefault();
		if (null == camera) {
			IJ.log("No webcam detected");
			return;
		}
		Prefs.set("Cam.newImage", false);

		if (!showDialog())
			return;
		camera = Webcam.getWebcams().get(camID);

		if (null != camera) {
			Dimension[] sizes = camera.getViewSizes();
			Dimension s1 = sizes[sizes.length - 1];

			if (customSize && (width > 0) && (height > 0)) {
				Dimension[] customSizes = new Dimension[1];
				customSizes[0] = new Dimension(width, height);
				camera.setCustomViewSizes(customSizes);
				s1 = customSizes[0];
			}

			camera.setViewSize(s1);
			camera.open();
			ip = new ColorProcessor(s1.width, s1.height);
			imp = new ImagePlus("Live (press Escape to grab)", ip);

			Calibration cal = imp.getCalibration();
			cal.setUnit(unit);
			cal.pixelWidth = calib;
			cal.pixelHeight = calib;

			WindowManager.addWindow(imp.getWindow());

			imp.show();
			image = camera.getImage();

			long frames = 0;
			double frameRate;
			long currentTime, initialTime = Calendar.getInstance().getTimeInMillis(), diff;
			String framerateString;

			while (!(IJ.escapePressed() || null == imp.getWindow())) {
				if (camera.isImageNew()) {
					image = camera.getImage();
					imp2 = new ImagePlus("tmp", image);
					ip = imp2.getProcessor();
					imp.setProcessor(ip);

					if (displayFPS) {
						frames++;
						currentTime = Calendar.getInstance().getTimeInMillis();
						diff = currentTime - initialTime;
						if (diff > 0) {
							frameRate = (double) frames * 1000;
							frameRate /= diff;
						} else
							frameRate = 0;
						framerateString = String.format("%.1f fps", frameRate);
						IJ.showStatus(framerateString);
					}

					imp.updateAndDraw();

					Prefs.set("Cam.newImage", true);

					if (grab)
						break;
				}
			}

			Prefs.set("Webcam.width", width);
			Prefs.set("Webcam.height", height);
			Prefs.set("Webcam.customSize", customSize);
			Prefs.set("Webcam.displayFPS", displayFPS);

			cal = imp.getCalibration();

			if ((cal.pixelWidth != calib) || (cal.getUnit() != unit)) {
				if (IJ.showMessageWithCancel("Preferences", "Calibration changed, replace?")) {
					Prefs.set("Webcam.calib", cal.pixelWidth);
					Prefs.set("Webcam.calUnit", cal.getUnit());

				}
			} else {
				Prefs.set("Webcam.calib", calib);
				Prefs.set("Webcam.calUnit", unit);
			}

			imp.setTitle("Snap");
			camera.close();
		}
	}

	boolean showDialog() {
		int n = 0;
		String[] cameraNames = new String[Webcam.getWebcams().size()];

		for (Webcam c : Webcam.getWebcams()) {
			cameraNames[n] = c.getName();
			n++;
		}

		customSize = (boolean) Prefs.get("Webcam.customSize", false);
		width = (int) Prefs.get("Webcam.width", width);
		height = (int) Prefs.get("Webcam.height", height);
		calib = (float) Prefs.get("Webcam.calib", 1.0);
		unit = (String) Prefs.get("Webcam.calUnit", "\u00B5m");
		displayFPS = (boolean) Prefs.get("Webcam.displayFPS", displayFPS);

		GenericDialog gd = new GenericDialog("IJ webcam plugin...");
		gd.addChoice("Camera name", cameraNames, cameraNames[0]);
		gd.addCheckbox("Show FPS in status bar", displayFPS);
		gd.addCheckbox("Grab and return", false);
		gd.addCheckbox("Custom size", customSize);
		gd.addNumericField("Width", width, 0, 5, "pixels");
		gd.addNumericField("Height", height, 0, 5, "pixels");
		gd.addMessage("Calibration");
		gd.addStringField("Unit", unit);
		gd.addNumericField("Pixel_size", calib, 8, 12, "units/px");
		gd.showDialog();
		if (gd.wasCanceled())
			return false;
		camID = (int) gd.getNextChoiceIndex();
		displayFPS = gd.getNextBoolean();
		grab = (boolean) gd.getNextBoolean();
		customSize = (boolean) gd.getNextBoolean();
		width = (int) gd.getNextNumber();
		height = (int) gd.getNextNumber();
		unit = (String) gd.getNextString();
		calib = (float) gd.getNextNumber();
		return true;
	}

}
