/*
 * Decompiled with CFR 0.152.
 */
package com.github.sarxos.webcam;

import com.github.sarxos.webcam.WebcamDevice;
import com.github.sarxos.webcam.WebcamDiscoveryListener;
import com.github.sarxos.webcam.WebcamDiscoveryService;
import com.github.sarxos.webcam.WebcamDriver;
import com.github.sarxos.webcam.WebcamDriverUtils;
import com.github.sarxos.webcam.WebcamEvent;
import com.github.sarxos.webcam.WebcamEventType;
import com.github.sarxos.webcam.WebcamException;
import com.github.sarxos.webcam.WebcamImageTransformer;
import com.github.sarxos.webcam.WebcamListener;
import com.github.sarxos.webcam.WebcamLock;
import com.github.sarxos.webcam.WebcamShutdownHook;
import com.github.sarxos.webcam.WebcamUpdater;
import com.github.sarxos.webcam.ds.buildin.WebcamDefaultDriver;
import com.github.sarxos.webcam.ds.cgt.WebcamCloseTask;
import com.github.sarxos.webcam.ds.cgt.WebcamDisposeTask;
import com.github.sarxos.webcam.ds.cgt.WebcamOpenTask;
import com.github.sarxos.webcam.ds.cgt.WebcamReadBufferTask;
import com.github.sarxos.webcam.ds.cgt.WebcamReadImageTask;
import com.github.sarxos.webcam.util.ImageUtils;
import java.awt.Dimension;
import java.awt.image.BufferedImage;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Webcam {
    private static final Logger LOG = LoggerFactory.getLogger(Webcam.class);
    private static final List<String> DRIVERS_LIST = new ArrayList<String>();
    private static final List<Class<?>> DRIVERS_CLASS_LIST = new ArrayList();
    private static final List<WebcamDiscoveryListener> DISCOVERY_LISTENERS = Collections.synchronizedList(new ArrayList());
    private static WebcamDriver driver = null;
    private static volatile WebcamDiscoveryService discovery = null;
    private static boolean deallocOnTermSignal = false;
    private static boolean autoOpen = false;
    private List<WebcamListener> listeners = Collections.synchronizedList(new ArrayList());
    private List<Dimension> customSizes = new ArrayList<Dimension>();
    private WebcamShutdownHook hook = null;
    private WebcamDevice device = null;
    private AtomicBoolean open = new AtomicBoolean(false);
    private AtomicBoolean disposed = new AtomicBoolean(false);
    private volatile boolean asynchronous = false;
    private volatile double fps = 0.0;
    private WebcamUpdater updater = new WebcamUpdater(this);
    private volatile WebcamImageTransformer transformer = null;
    private WebcamLock lock = null;

    protected Webcam(WebcamDevice device) {
        if (device == null) {
            throw new IllegalArgumentException("Webcam device cannot be null");
        }
        this.device = device;
        this.lock = new WebcamLock(this);
    }

    public boolean open() {
        return this.open(false);
    }

    public boolean open(boolean async) {
        if (this.open.compareAndSet(false, true)) {
            assert (this.updater != null);
            assert (this.lock != null);
            this.lock.lock();
            WebcamOpenTask task = new WebcamOpenTask(driver, this.device);
            try {
                task.open();
            }
            catch (InterruptedException e) {
                this.lock.unlock();
                this.open.set(false);
                LOG.debug("Thread has been interrupted in the middle of webcam opening process!", (Throwable)e);
                return false;
            }
            catch (WebcamException e) {
                this.lock.unlock();
                this.open.set(false);
                throw e;
            }
            LOG.debug("Webcam is now open {}", (Object)this.getName());
            this.asynchronous = async;
            if (async) {
                this.updater.start();
            }
            this.hook = new WebcamShutdownHook(this);
            Runtime.getRuntime().addShutdownHook(this.hook);
            WebcamEvent we = new WebcamEvent(WebcamEventType.OPEN, this);
            for (WebcamListener l : this.getWebcamListeners()) {
                try {
                    l.webcamOpen(we);
                }
                catch (Exception e) {
                    LOG.error(String.format("Notify webcam open, exception when calling listener %s", l.getClass()), (Throwable)e);
                }
            }
        } else {
            LOG.debug("Webcam is already open {}", (Object)this.getName());
        }
        return true;
    }

    public boolean close() {
        if (this.open.compareAndSet(true, false)) {
            assert (this.updater != null);
            assert (this.lock != null);
            WebcamCloseTask task = new WebcamCloseTask(driver, this.device);
            try {
                task.close();
            }
            catch (InterruptedException e) {
                this.open.set(true);
                LOG.debug("Thread has been interrupted before webcam was closed!", (Throwable)e);
                return false;
            }
            catch (WebcamException e) {
                this.open.set(true);
                throw e;
            }
            this.lock.unlock();
            if (this.asynchronous) {
                this.updater.stop();
            }
            this.removeShutdownHook();
            WebcamEvent we = new WebcamEvent(WebcamEventType.CLOSED, this);
            for (WebcamListener l : this.getWebcamListeners()) {
                try {
                    l.webcamClosed(we);
                }
                catch (Exception e) {
                    LOG.error(String.format("Notify webcam closed, exception when calling %s listener", l.getClass()), (Throwable)e);
                }
            }
        } else {
            LOG.debug("Webcam is already closed {}", (Object)this.getName());
        }
        return true;
    }

    public WebcamDevice getDevice() {
        assert (this.device != null);
        return this.device;
    }

    protected void dispose() {
        assert (this.disposed != null);
        assert (this.open != null);
        assert (driver != null);
        assert (this.device != null);
        assert (this.listeners != null);
        if (!this.disposed.compareAndSet(false, true)) {
            return;
        }
        this.open.set(false);
        LOG.info("Disposing webcam {}", (Object)this.getName());
        WebcamDisposeTask task = new WebcamDisposeTask(driver, this.device);
        try {
            task.dispose();
        }
        catch (InterruptedException e) {
            LOG.error("Processor has been interrupted before webcam was disposed!", (Throwable)e);
            return;
        }
        WebcamEvent we = new WebcamEvent(WebcamEventType.DISPOSED, this);
        for (WebcamListener l : this.listeners) {
            try {
                l.webcamClosed(we);
                l.webcamDisposed(we);
            }
            catch (Exception e) {
                LOG.error(String.format("Notify webcam disposed, exception when calling %s listener", l.getClass()), (Throwable)e);
            }
        }
        this.removeShutdownHook();
        LOG.debug("Webcam disposed {}", (Object)this.getName());
    }

    private void removeShutdownHook() {
        if (this.hook != null) {
            try {
                Runtime.getRuntime().removeShutdownHook(this.hook);
            }
            catch (IllegalStateException e) {
                LOG.trace("Shutdown in progress, cannot remove hook");
            }
        }
    }

    protected BufferedImage transform(BufferedImage image) {
        WebcamImageTransformer tr;
        if (image != null && (tr = this.getImageTransformer()) != null) {
            return tr.transform(image);
        }
        return image;
    }

    public boolean isOpen() {
        return this.open.get();
    }

    public Dimension getViewSize() {
        return this.device.getResolution();
    }

    public Dimension[] getViewSizes() {
        return this.device.getResolutions();
    }

    public void setCustomViewSizes(Dimension[] sizes) {
        assert (this.customSizes != null);
        if (sizes == null) {
            this.customSizes.clear();
            return;
        }
        this.customSizes = Arrays.asList(sizes);
    }

    public Dimension[] getCustomViewSizes() {
        assert (this.customSizes != null);
        return this.customSizes.toArray(new Dimension[this.customSizes.size()]);
    }

    public void setViewSize(Dimension size) {
        if (size == null) {
            throw new IllegalArgumentException("Resolution cannot be null!");
        }
        if (this.open.get()) {
            throw new IllegalStateException("Cannot change resolution when webcam is open, please close it first");
        }
        Dimension current = this.getViewSize();
        if (current != null && current.width == size.width && current.height == size.height) {
            return;
        }
        Dimension[] predefined = this.getViewSizes();
        Dimension[] custom = this.getCustomViewSizes();
        assert (predefined != null);
        assert (custom != null);
        boolean ok = false;
        for (Dimension d : predefined) {
            if (d.width != size.width || d.height != size.height) continue;
            ok = true;
            break;
        }
        if (!ok) {
            for (Dimension d : custom) {
                if (d.width != size.width || d.height != size.height) continue;
                ok = true;
                break;
            }
        }
        if (!ok) {
            StringBuilder sb = new StringBuilder("Incorrect dimension [");
            sb.append(size.width).append("x").append(size.height).append("] ");
            sb.append("possible ones are ");
            for (Dimension d : predefined) {
                sb.append("[").append(d.width).append("x").append(d.height).append("] ");
            }
            for (Dimension d : custom) {
                sb.append("[").append(d.width).append("x").append(d.height).append("] ");
            }
            throw new IllegalArgumentException(sb.toString());
        }
        LOG.debug("Setting new resolution {}x{}", (Object)size.width, (Object)size.height);
        this.device.setResolution(size);
    }

    public BufferedImage getImage() {
        if (!this.isReady()) {
            return null;
        }
        long t1 = 0L;
        long t2 = 0L;
        assert (this.updater != null);
        if (this.asynchronous) {
            return this.updater.getImage();
        }
        t1 = System.currentTimeMillis();
        BufferedImage image = this.transform(new WebcamReadImageTask(driver, this.device).getImage());
        t2 = System.currentTimeMillis();
        if (image == null) {
            return null;
        }
        this.fps = this.device instanceof WebcamDevice.FPSSource ? ((WebcamDevice.FPSSource)((Object)this.device)).getFPS() : (4.0 * this.fps + (double)(1000L / (t2 - t1 + 1L))) / 5.0;
        this.updater.notifyWebcamImageObtained(this, image);
        return image;
    }

    public boolean isImageNew() {
        assert (this.updater != null);
        if (this.asynchronous) {
            return this.updater.isImageNew();
        }
        return true;
    }

    protected double getFPS() {
        assert (this.updater != null);
        if (this.asynchronous) {
            return this.updater.getFPS();
        }
        return this.fps;
    }

    public ByteBuffer getImageBytes() {
        if (!this.isReady()) {
            return null;
        }
        assert (driver != null);
        assert (this.device != null);
        if (this.device instanceof WebcamDevice.BufferAccess) {
            return new WebcamReadBufferTask(driver, this.device).getBuffer();
        }
        BufferedImage image = this.getImage();
        if (image != null) {
            return ByteBuffer.wrap(ImageUtils.toRawByteArray(image));
        }
        return null;
    }

    private boolean isReady() {
        assert (this.disposed != null);
        assert (this.open != null);
        if (this.disposed.get()) {
            LOG.warn("Cannot get image, webcam has been already disposed");
            return false;
        }
        if (!this.open.get()) {
            if (autoOpen) {
                this.open();
            } else {
                return false;
            }
        }
        return true;
    }

    public static List<Webcam> getWebcams() throws WebcamException {
        try {
            return Webcam.getWebcams(Long.MAX_VALUE);
        }
        catch (TimeoutException e) {
            throw new RuntimeException(e);
        }
    }

    public static List<Webcam> getWebcams(long timeout) throws TimeoutException, WebcamException {
        if (timeout < 0L) {
            throw new IllegalArgumentException(String.format("Timeout cannot be negative (%d)", timeout));
        }
        return Webcam.getWebcams(timeout, TimeUnit.MILLISECONDS);
    }

    public static synchronized List<Webcam> getWebcams(long timeout, TimeUnit tunit) throws TimeoutException, WebcamException {
        if (timeout < 0L) {
            throw new IllegalArgumentException(String.format("Timeout cannot be negative (%d)", timeout));
        }
        if (tunit == null) {
            throw new IllegalArgumentException("Time unit cannot be null!");
        }
        WebcamDiscoveryService discovery = Webcam.getDiscoveryService();
        assert (discovery != null);
        List<Webcam> webcams = discovery.getWebcams(timeout, tunit);
        if (!discovery.isRunning()) {
            discovery.start();
        }
        return webcams;
    }

    public static Webcam getDefault() throws WebcamException {
        try {
            return Webcam.getDefault(Long.MAX_VALUE);
        }
        catch (TimeoutException e) {
            throw new RuntimeException(e);
        }
    }

    public static Webcam getDefault(long timeout) throws TimeoutException, WebcamException {
        if (timeout < 0L) {
            throw new IllegalArgumentException(String.format("Timeout cannot be negative (%d)", timeout));
        }
        return Webcam.getDefault(timeout, TimeUnit.MILLISECONDS);
    }

    public static Webcam getDefault(long timeout, TimeUnit tunit) throws TimeoutException, WebcamException {
        if (timeout < 0L) {
            throw new IllegalArgumentException(String.format("Timeout cannot be negative (%d)", timeout));
        }
        if (tunit == null) {
            throw new IllegalArgumentException("Time unit cannot be null!");
        }
        List<Webcam> webcams = Webcam.getWebcams(timeout, tunit);
        assert (webcams != null);
        if (!webcams.isEmpty()) {
            return webcams.get(0);
        }
        LOG.warn("No webcam has been detected!");
        return null;
    }

    public String getName() {
        assert (this.device != null);
        return this.device.getName();
    }

    public String toString() {
        return String.format("Webcam %s", this.getName());
    }

    public boolean addWebcamListener(WebcamListener l) {
        if (l == null) {
            throw new IllegalArgumentException("Webcam listener cannot be null!");
        }
        assert (this.listeners != null);
        return this.listeners.add(l);
    }

    public WebcamListener[] getWebcamListeners() {
        assert (this.listeners != null);
        return this.listeners.toArray(new WebcamListener[this.listeners.size()]);
    }

    public int getWebcamListenersCount() {
        assert (this.listeners != null);
        return this.listeners.size();
    }

    public boolean removeWebcamListener(WebcamListener l) {
        assert (this.listeners != null);
        return this.listeners.remove(l);
    }

    public static synchronized WebcamDriver getDriver() {
        if (driver != null) {
            return driver;
        }
        if (driver == null) {
            driver = WebcamDriverUtils.findDriver(DRIVERS_LIST, DRIVERS_CLASS_LIST);
        }
        if (driver == null) {
            driver = new WebcamDefaultDriver();
        }
        LOG.info("{} capture driver will be used", (Object)driver.getClass().getSimpleName());
        return driver;
    }

    public static synchronized void setDriver(WebcamDriver wd) {
        if (wd == null) {
            throw new IllegalArgumentException("Webcam driver cannot be null!");
        }
        LOG.debug("Setting new capture driver {}", (Object)wd);
        Webcam.resetDriver();
        driver = wd;
    }

    public static synchronized void setDriver(Class<? extends WebcamDriver> driverClass) {
        if (driverClass == null) {
            throw new IllegalArgumentException("Webcam driver class cannot be null!");
        }
        Webcam.resetDriver();
        try {
            driver = driverClass.newInstance();
        }
        catch (InstantiationException e) {
            throw new WebcamException(e);
        }
        catch (IllegalAccessException e) {
            throw new WebcamException(e);
        }
    }

    public static synchronized void resetDriver() {
        DRIVERS_LIST.clear();
        if (discovery != null) {
            discovery.shutdown();
            discovery = null;
        }
        driver = null;
    }

    public static void registerDriver(Class<? extends WebcamDriver> clazz) {
        if (clazz == null) {
            throw new IllegalArgumentException("Webcam driver class to register cannot be null!");
        }
        DRIVERS_CLASS_LIST.add(clazz);
        Webcam.registerDriver(clazz.getCanonicalName());
    }

    public static void registerDriver(String clazzName) {
        if (clazzName == null) {
            throw new IllegalArgumentException("Webcam driver class name to register cannot be null!");
        }
        DRIVERS_LIST.add(clazzName);
    }

    public static void setHandleTermSignal(boolean on) {
        if (on) {
            LOG.warn("Automated deallocation on TERM signal is now enabled! Make sure to not use it in production!");
        }
        deallocOnTermSignal = on;
    }

    public static boolean isHandleTermSignal() {
        return deallocOnTermSignal;
    }

    public static void setAutoOpenMode(boolean on) {
        autoOpen = on;
    }

    public static boolean isAutoOpenMode() {
        return autoOpen;
    }

    public static boolean addDiscoveryListener(WebcamDiscoveryListener l) {
        if (l == null) {
            throw new IllegalArgumentException("Webcam discovery listener cannot be null!");
        }
        return DISCOVERY_LISTENERS.add(l);
    }

    public static WebcamDiscoveryListener[] getDiscoveryListeners() {
        return DISCOVERY_LISTENERS.toArray(new WebcamDiscoveryListener[DISCOVERY_LISTENERS.size()]);
    }

    public static boolean removeDiscoveryListener(WebcamDiscoveryListener l) {
        return DISCOVERY_LISTENERS.remove(l);
    }

    public static synchronized WebcamDiscoveryService getDiscoveryService() {
        if (discovery == null) {
            discovery = new WebcamDiscoveryService(Webcam.getDriver());
        }
        return discovery;
    }

    public static synchronized WebcamDiscoveryService getDiscoveryServiceRef() {
        return discovery;
    }

    public WebcamImageTransformer getImageTransformer() {
        return this.transformer;
    }

    public void setImageTransformer(WebcamImageTransformer transformer) {
        this.transformer = transformer;
    }

    public WebcamLock getLock() {
        return this.lock;
    }
}

